/**
* \file: AlsaAudioCommon.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: I. Hayashi / ADITJ/SW / ihayashi@jp.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef AAUTO_ALSAAUDIOCOMMON_H
#define AAUTO_ALSAAUDIOCOMMON_H

#include <atomic>
#include <string>
#include <alsa/asoundlib.h>
/* header to use sync mechanism */
#include <aauto/util/Mutex.h>


/* Google Integration guide notes AAC-lc format is:
 * RAW AAC frame without header, decoded buffer size in one frame:
 *      1024 samples for 16kHz,
 *      2048 samples for 48kHz.
 */
#define SAMPLES_IN_ONE_FRAME_FOR_16KHZ      1024    // decoded buffer size in one frame for 16kHz
#define SAMPLES_IN_ONE_FRAME_FOR_48KHZ      2048    // decoded buffer size in one frame for 48kHz


namespace adit { namespace aauto
{

class AlsaAudioCommon
{
public :
    AlsaAudioCommon(bool inDumpConfiguration, bool inVerboseLogging);
    ~AlsaAudioCommon();

    enum AlsaAudioStates {
        /* set at SetupPCM() automatically */
        AlsaAudioStart = 0,
        /* For playback, wait for all queued work items to be played */
        AlsaAudioDrain,
        /* For playback, stop playback after current work item was played */
        AlsaAudioDrop,
        /* For playback and capture, stop immediately streaming.
        Default state. */
        AlsaAudioStop
    };

    /* to set configuration */
    void SetStreamName(const char *audio_stream_name);
    void SetDir (snd_pcm_stream_t dir_);
    void SetFormat (snd_pcm_format_t format_);
    void SetChannels(unsigned int channels_);
    void SetRate (unsigned int rate_);
    void SetInitTout(int tout_);
    void SetThresholdMs(unsigned int threshold_ms_);
    void SetIdealPeriodMs(unsigned int ideal_period_ms_);
    void SetBufferIdealPeriods (int buffer_ideal_periods_);
    void SetPrefillMs(int prefill_ms_);
    void SetRecoverDelayMs(unsigned int recover_delay_ms_);

    snd_pcm_uframes_t GetIdealPeriodMs();
    snd_pcm_uframes_t GetPrefillMs();
    size_t GetIdealPeriodBytes();

    /**
     * @brief Calculate the threshold buffer byte size based on the parameter set at SetupPCM().
     *
     * @param samples_per_frame_ Google definition: number of samples in one frame.
     * return threshold buffer size (>0) if success. Otherwise 0.
     */
    int GetThresholdBufferSize(unsigned int samples_per_frame_);

    /**
     * @brief Open ALSA device (snd_pcm_*) and set parameter
     *
     * @param pcmname the name of the ALSA device which should be open
     */
    int SetupPCM(const char *pcmname);
    void ReleasePCM();

    /* buffer has to contain all channels interleaved in the configured format */
    int ReadWrite(void *buffer, size_t len);

    int StopStreaming();
    int Delay(snd_pcm_sframes_t *delay);

    /* use to set the state from the outside */
    void setAlsaAudioState(AlsaAudioStates inState) {
        Autolock l(&mLock);
        mStreamState = inState;
    };
    AlsaAudioStates getAlsaAudioState() {
        Autolock l(&mLock);
        return mStreamState;
    };

    /* Controls if the divisor is zero and calculates the remainder after division.
     * @param inDividen is the dividend
     * @param inDivisor is the divisor
     * @return If success, the remainder after division.
     *         Returns 0 if the divisor is 0. */
    template<class T>
    T safe_modulo(T inDividen, T inDivisor) {
        return ((inDivisor>0)?(inDividen % inDivisor):0);
    }

    // private section
private:
    int CreateBuffers ();
    void ReleaseBuffers ();

    int WriteSilence(const snd_pcm_uframes_t silence);
    int WaitPCM();
    int RwBuffer(void *buffer, snd_pcm_uframes_t xfer_size_);
    int Recover();

    /* configured by using application */
    snd_pcm_stream_t dir;
    snd_pcm_format_t format;/*format*/
    unsigned int channels; /* number of channels */
    unsigned int rate;    /*any, e.g. 8000-192000*/
    unsigned int ideal_period_ms;
    int buffer_ideal_periods;
    int prefill_ms;
    int recover_delay_ms;

    const char* streamName;

    /* threshold time in millisecond configured by Application */
    unsigned int threshold_ms;
    /* number of frames for the threshold */
    unsigned int threshold_frames;
    /* number of bytes for the threshold */
    int threshold_buffer_size;

    /* internally configured */
    int init_tout;  /*timeout on snd_pcm_wait() until stream is active*/
    snd_pcm_uframes_t startup_xfer; /*sum of read/written frames- count up to buffer-size to detect device activity*/
    snd_pcm_t *pcm;
    snd_pcm_access_t access;
    snd_pcm_uframes_t ideal_period_size;
    snd_pcm_uframes_t silence_prefill;

    snd_pcm_uframes_t buffer_size;
    snd_pcm_uframes_t period_size;

    unsigned int bytes_per_frame;

    void *silence_buffer;

    AlsaAudioStates mStreamState;
    bool mStreamStarted;
    bool mStreamStopped;
    bool mStreamBroken;
    bool mSetupPCM;
    bool wait_done;

    bool dump_config;
    bool verbose;
    /* Mutex to lock the state change (set/get) */
    Mutex mLock;
};

} } /* namespace adit { namespace aauto */

#endif /* AAUTO_ALSAAUDIOCOMMON_H */
